/*-----------------------------------------------------------------------------
 * AtEthercat.h             header file
 * Copyright                acontis technologies GmbH, Weingarten, Germany
 * Response                 Stefan Zintgraf
 * Description              interface to the ethercat master
 *---------------------------------------------------------------------------*/

#ifndef INC_ATETHERCAT
#define INC_ATETHERCAT

/*-INCLUDE-------------------------------------------------------------------*/
#ifndef INC_ECTYPE
#include "EcType.h"
#endif
#ifndef INC_ECERROR
#include "EcError.h"
#endif
#ifndef INC_ECINTERFACECOMMON
#include "EcInterfaceCommon.h"
#endif
#ifndef INC_ECOS
#include "EcOs.h"
#endif
#include "EcVersion.h"
#ifndef INC_ECLINK
#include "EcLink.h"
#endif
#ifndef INC_ETHERNETSERVICES
#include "EthernetServices.h"
#endif


/*-PACK SETTINGS-------------------------------------------------------------*/
#if defined  __GNUC__   /* GNU */

#define EC_PACKED __attribute__((aligned(4), packed))

#if (defined __RCX__)
#pragma pack (4)
#endif

#elif defined _DIAB_TOOL

#define EC_PACKED __attribute__((aligned(4), packed))

#elif defined _MSC_VER  /* MICROSOFT */

#define EC_PACKED
#pragma pack (push, 4)
#pragma warning(disable:4514)

#elif defined __MET__

#define EC_PACKED
/*#pragma pack (push, 4)*/
#pragma Push_align_members(4)

#else
#error Please check pack settings for this compiler!!!
#endif


/*-DEFINES/MACROS------------------------------------------------------------*/

#define ATECAT_VALIDATION_PATTERN 0xAECAAECA
/* EtherCat specific control codes */
#define EC_IOCTL_GENERIC                            0x00000000
#define EC_IOCTL_PROCDATA                           0x00010000
#define EC_IOCTL_MBOX                               0x00020000
#define EC_IOCTL_DC                                 0x00030000
#define EC_IOCTL_DCL                                0x00040000
#define EC_IOCTL_SB                                 0x00050000
#define EC_IOCTL_HC                                 0x00060000

#define EC_IOCTL_DCM                                0x00070000

#define EC_IOCTL_PRIVATE                            0x00FF0000  /* private, internal IOCTL values */

#define EC_IOCTL_GETSTATE                           (EC_IOCTL_GENERIC |  1)
#define EC_IOCTL_REGISTERCLIENT                     (EC_IOCTL_GENERIC |  2)
#define EC_IOCTL_UNREGISTERCLIENT                   (EC_IOCTL_GENERIC |  3)
#define EC_IOCTL_ISLINK_CONNECTED                   (EC_IOCTL_GENERIC |  6)
#define EC_IOCTL_SET_PHYS_MBX_POLLING_PERIOD        (EC_IOCTL_GENERIC |  7)
#define EC_IOCTL_SET_CYC_ERROR_NOTIFY_MASK          (EC_IOCTL_GENERIC |  8)
#define EC_IOCTL_SET_SLAVE_STATE_UPDATE_TIMEOUT     (EC_IOCTL_GENERIC |  9)
#define EC_IOCTL_LINKLAYER_DBG_MSG                  (EC_IOCTL_GENERIC | 10)
#define EC_IOCTL_SET_COE_DBG_LEVEL                  (EC_IOCTL_GENERIC | 11)
#define EC_IOCTL_UPDATE_ALL_SLAVE_STATE             (EC_IOCTL_GENERIC | 12)
#define EC_IOCTL_RESET_SLAVE                        (EC_IOCTL_GENERIC | 13)
#define EC_IOCTL_SLAVE_LINKMESSAGES                 (EC_IOCTL_GENERIC | 14)
#define EC_IOCTL_GET_CYCLIC_CONFIG_INFO             (EC_IOCTL_GENERIC | 15)
#define EC_IOCTL_GET_LINKLAYER_MODE                 (EC_IOCTL_GENERIC | 16)
#define EC_IOCTL_UPDATE_ALL_SLAVE_STATE_NOWAIT      (EC_IOCTL_GENERIC | 17)
#define EC_IOCTL_SET_FOE_DBG_LEVEL                  (EC_IOCTL_GENERIC | 18)
#define EC_IOCTL_INITIATE_UPDATE_ALL_SLAVE_STATE    (EC_IOCTL_GENERIC | 19)
#define EC_IOCTL_SET_SOE_DBG_LEVEL                  (EC_IOCTL_GENERIC | 20)
#define EC_IOCTL_ONLY_PROCESS_DATA_IN_IMAGE         (EC_IOCTL_GENERIC | 21)

#define EC_IOCTL_REGISTER_CYCFRAME_RX_CB            (EC_IOCTL_GENERIC | 22)
#define EC_IOCTL_SET_PD_OFFSET_COMPAT_MODE          (EC_IOCTL_GENERIC | 23)

#define EC_IOCTL_IS_MAIN_LINK_CONNECTED             (EC_IOCTL_GENERIC | 24)
#define EC_IOCTL_IS_RED_LINK_CONNECTED              (EC_IOCTL_GENERIC | 25)

    /* Memory Provider */
#define EC_IOCTL_GET_PDMEMORYSIZE                   (EC_IOCTL_GENERIC | 40)
#define EC_IOCTL_REGISTER_PDMEMORYPROVIDER          (EC_IOCTL_GENERIC | 41)

#define EC_IOCTL_FORCE_BROADCAST_DESTINATION        (EC_IOCTL_GENERIC | 42)

    /* Slave Statistics Retrieval */
#if (defined INCLUDE_SLAVE_STATISTICS)
#define EC_IOCTL_SET_SLVSTAT_PERIOD                 (EC_IOCTL_GENERIC | 43)
#define EC_IOCTL_FORCE_SLVSTAT_COLLECTION           (EC_IOCTL_GENERIC | 44)
#define EC_IOCTL_GET_SLVSTATISTICS                  (EC_IOCTL_GENERIC | 45)
#define EC_IOCTL_CLR_SLVSTATISTICS                  (EC_IOCTL_GENERIC | 46)
#endif

    /* Distributed Clocks (DC) */                   
#define EC_IOCTL_DC_ENABLE                          (EC_IOCTL_DC |  1)
#define EC_IOCTL_DC_DISABLE                         (EC_IOCTL_DC |  2)

#define EC_IOCTL_REG_DC_SLV_SYNC_NTFY               (EC_IOCTL_DC |  3)
#define EC_IOCTL_UNREG_DC_SLV_SYNC_NTFY             (EC_IOCTL_DC |  4)
#define EC_IOCTL_DC_SLV_SYNC_STATUS_GET             (EC_IOCTL_DC |  5)
#define EC_IOCTL_DC_SLV_SYNC_DEVLIMIT_SET           (EC_IOCTL_DC |  6)
#define EC_IOCTL_DC_SLV_SYNC_DEVLIMIT_GET           (EC_IOCTL_DC |  7)
#define EC_IOCTL_DC_SLAVESYNCDISABLE                (EC_IOCTL_DC |  8)      /* not supported */
#define EC_IOCTL_DC_PDM_CYCLES_SET                  (EC_IOCTL_DC |  9)      /* not supported */
#define EC_IOCTL_DC_PDM_CYCLES_GET                  (EC_IOCTL_DC | 10)      /* not supported */
#define EC_IOCTL_DC_SLV_SYNC_RESTART                (EC_IOCTL_DC | 11)
#define EC_IOCTL_DC_SLV_SYNC_SETTLETIME_SET         (EC_IOCTL_DC | 12)
#define EC_IOCTL_DC_SLV_SYNC_SETTLETIME_GET         (EC_IOCTL_DC | 13)
#define EC_IOCTL_DC_CONFIGURE_BURST                 (EC_IOCTL_DC | 14)
#define EC_IOCTL_DC_SLV_SYNC_RESTART_WSYNC          (EC_IOCTL_DC | 15)

#define EC_IOCTL_DC_SHIFT_SYSTIME                   (EC_IOCTL_DC | 16)
#define EC_IOCTL_DC_SETSYNCSTARTOFFSET              (EC_IOCTL_DC | 17)
#define EC_IOCTL_DC_SLV_SYNCPULSE_RESTART           (EC_IOCTL_DC | 18)

    /* DC Latching (DCL) */
#define EC_IOCTL_DC_LATCH_ENABLE                    (EC_IOCTL_DCL | 1)
#define EC_IOCTL_DC_LATCH_DISABLE                   (EC_IOCTL_DCL | 2)
#define EC_IOCTL_REG_DC_LATCH_NTFY                  (EC_IOCTL_DCL | 3)
#define EC_IOCTL_UNREG_DC_LATCH_NTFY                (EC_IOCTL_DCL | 4)
#define EC_IOCTL_DC_LATCH_GETLTIMVALS               (EC_IOCTL_DCL | 5)
#define EC_IOCTL_DC_LATCH_SETMODE                   (EC_IOCTL_DCL | 6)
#define EC_IOCTL_DC_LATCH_GETMODE                   (EC_IOCTL_DCL | 7)

    /* Scan Bus (SB) */
#define EC_IOCTL_SB_RESTART                         (EC_IOCTL_SB | 1)
#define EC_IOCTL_SB_STATUS_GET                      (EC_IOCTL_SB | 2)
#define EC_IOCTL_SB_SET_BUSCNF_VERIFY               (EC_IOCTL_SB | 3)
#define EC_IOCTL_SB_SET_BUSCNF_VERIFY_PROP          (EC_IOCTL_SB | 4)
#define EC_IOCTL_SB_BUSCNF_GETSLAVE_INFO            (EC_IOCTL_SB | 5)
#define EC_IOCTL_SB_BUSCNF_GETSLAVE_INFO_EEP        (EC_IOCTL_SB | 6)
#define EC_IOCTL_SB_ENABLE                          (EC_IOCTL_SB | 7)
#define EC_IOCTL_SB_DISABLE                         (EC_IOCTL_SB | 8)
#define EC_IOCTL_SB_BUSCNF_GETSLAVE_INFO_EX         (EC_IOCTL_SB | 9)

#define EC_IOCTL_SLV_ALIAS_ENABLE                   (EC_IOCTL_SB |10)
#define EC_IOCTL_SB_MEMORYPOOLRESIZE                (EC_IOCTL_SB |11)

    /* Hot Connect (HC) */
#define EC_IOCTL_HC_SETMODE                         (EC_IOCTL_HC | 1)
#define EC_IOCTL_HC_GETMODE                         (EC_IOCTL_HC | 2)
#define EC_IOCTL_HC_CONFIGURETIMEOUTS               (EC_IOCTL_HC | 3)

    /* DC Master Sync (DCM) */
#if (defined INCLUDE_MASTER_SYNC)
    
#define EC_IOCTL_DCM_REGISTER_TIMESTAMP             (EC_IOCTL_DCM |  1)
#define EC_IOCTL_DCM_UNREGISTER_TIMESTAMP           (EC_IOCTL_DCM |  2)
#define EC_IOCTL_DCM_REGISTER_STARTSO_CALLBACK      (EC_IOCTL_DCM |  3)

#endif

    /* private (PRIVATE) */
#define EC_IOCTL_SET_FRAME_LOSS_SIMULATION          (EC_IOCTL_PRIVATE | 1)
#define EC_IOCTL_SET_RXFRAME_LOSS_SIMULATION        (EC_IOCTL_PRIVATE | 2)
#define EC_IOCTL_SET_TXFRAME_LOSS_SIMULATION        (EC_IOCTL_PRIVATE | 3)
#define EC_IOCTL_SET_SOFT_ASSERTIONS                (EC_IOCTL_PRIVATE | 4)
#define EC_IOCTL_SET_HARD_ASSERTIONS                (EC_IOCTL_PRIVATE | 5)

/* EtherCat CoE entry description value information bit definitions */
#define EC_COE_ENTRY_ObjAccess              0x01
#define EC_COE_ENTRY_ObjCategory            0x02
#define EC_COE_ENTRY_PdoMapping             0x04
#define EC_COE_ENTRY_UnitType               0x08
#define EC_COE_ENTRY_DefaultValue           0x10
#define EC_COE_ENTRY_MinValue               0x20
#define EC_COE_ENTRY_MaxValue               0x40

/* EtherCat CoE entry access bit definitions */
#define EC_COE_ENTRY_Access_R_PREOP         0x01
#define EC_COE_ENTRY_Access_R_SAFEOP        0x02
#define EC_COE_ENTRY_Access_R_OP            0x04
#define EC_COE_ENTRY_Access_W_PREOP         0x08
#define EC_COE_ENTRY_Access_W_SAFEOP        0x10
#define EC_COE_ENTRY_Access_W_OP            0x20

#define INVALID_SLAVE_ID                    ((EC_T_DWORD)0xFFFFFFFF)
#define INVALID_AUTO_INC_ADDR               ((EC_T_DWORD)0x1)


#define MAX_PROCESS_VAR_NAME_LEN            ((EC_T_DWORD)72) /* Maximum length of a process variable name */
                     
#define INSTANCE_MASTER_DEFAULT             0                /* first (default) master instance used by ecatXXXXX */
#define MAX_NUMOF_MASTER_INSTANCES          10               /* maximum number of master instances */


/*-TYPEDEFS/ENUMS------------------------------------------------------------*/

struct _EC_T_NOTIFYPARMS;

/* EtherCat callback function pointer */
typedef EC_T_DWORD (*EC_PF_NOTIFY)(                 EC_T_DWORD dwCode, 
                                                    struct _EC_T_NOTIFYPARMS* pParms    );

/* EtherCAT ioctl parameters */                     
typedef struct _EC_T_IOCTLOPARMS                    
{                                                   
    EC_T_BYTE*      pbyInBuf;                       /*< in  input data buffer */
    EC_T_DWORD      dwInBufSize;                    /*< in  size of input data buffer in byte */
    EC_T_BYTE*      pbyOutBuf;                      /*< out output data buffer */
    EC_T_DWORD      dwOutBufSize;                   /*< in  size of output data buffer in byte */
    EC_T_DWORD*     pdwNumOutData;                  /*< out number of output data bytes stored in output data buffer */
}EC_PACKED EC_T_IOCTLPARMS;                                  
                                                    
/* Client register parameters */                    
typedef struct _EC_T_REGISTERPARMS                  
{                                                   
    EC_T_VOID*      pCallerData;                    /*< in  used by all callback functions */
    EC_PF_NOTIFY    pfnNotify;                      /*< in  notify callback function pointer */
}/*EC_PACKED */EC_T_REGISTERPARMS; /* Packed function pointer causes error with VxWorks DIAB compiler */
                                                    
/* Client register result */                        
typedef struct _EC_T_REGISTERRESULTS                
{                                                   
    EC_T_DWORD      dwClntId;                       /*< out client ID */
    EC_T_BYTE*      pbyPDIn;                        /*< out incoming process data shared memory */
    EC_T_DWORD      dwPDInSize;                     /*< out size of the incoming process data area */
    EC_T_BYTE*      pbyPDOut;                       /*< out outgoing process data shared memory */
    EC_T_DWORD      dwPDOutSize;                    /*< out size of the outgoing process data area */
}EC_PACKED EC_T_REGISTERRESULTS;

/* Master initialization parameters */
typedef struct _EC_T_INITMASTERPARMS
{
    EC_T_DWORD          dwValPattern;                   /*< in  Validation pattern */
    EC_T_DWORD          dwVersion;                      /*< in  Version information */
    EC_T_VOID*          pvConfigParms;                  /*< in  obsolete */
    EC_T_VOID*          pvLinkParms;                    /*< in  Link Layer parameter */
    EC_T_OS_INIT_DESC*  pOsParms;                       /*< in  OS Layer parameter */
    EC_T_SYNCMODE       eSyncMode;                      /*< in  obsolete */
    EC_T_BOOL           bUseRedundancy;                 /*< in  Use redundancy device */
    EC_T_VOID*          pvLinkParmsRed;                 /*< in  Link Layer parameter for red device */
    EC_T_DWORD          dwVlanConfig;                   /*< in  VLAN support configuration 
                                                         *      Errr.rrrr|rrrr.rrrr|rPPP.IIII|IIII.IIII 
                                                         *      E=enable (1/0), P=Prio (3Bit), I=VLAN Id (12Bit) */
    
    EC_T_DWORD          dwReserved;                     /*< in  obsolete */
    EC_T_MASTER_CONFIG* pMasterConfig;                  /*< in  pointer to static master configuration */
    EC_T_MASTER_CONFIG  MasterConfig;                   /*< in  static master configuration */
}EC_PACKED EC_T_INITMASTERPARMS;

/* link layer debug message */                    
typedef struct _EC_T_LINKLAYER_DBG_MSG_DESC
{                                                   
    EC_T_BYTE       byEthTypeByte0;                 /*< in  Ethernet type byte 0 */
    EC_T_BYTE       byEthTypeByte1;                 /*< in  Ethernet type byte 0 */
    EC_T_CHAR*      szMsg;                          /*< in  message to send to link layer */
}EC_PACKED EC_T_LINKLAYER_DBG_MSG_DESC;                               

/*  CoE debug descriptor (obsolete, use MBX DBG DESC, for compatibility purposes only) */
typedef struct _EC_T_COE_DBG_DESC
{
    EC_T_BOOL   bEnable;                            /* EC_TRUE if enabled */
    EC_T_DWORD  dwCoETypeMask;                      /* Mask of CoE types to show */
}EC_PACKED EC_T_COE_DBG_DESC;

/*  Mailbox debug descriptor */
typedef struct _EC_T_MBX_DBG_DESC
{
    EC_T_BOOL   bEnable;                            /* EC_TRUE if enabled */
    EC_T_DWORD  dwMbxTypeMask;                      /* Mask of CoE types to show */
}EC_PACKED EC_T_MBX_DBG_DESC;

                                                    
/* Distributed clocks */
typedef struct _EC_T_DC_SYSTIME_ADD_OFFSET_DESC
{
    EC_T_DWORD      IsNegative;                     /*< in  System time offset is Negative value */
    EC_T_DWORD      dwOffset;                       /*< in  absolute system time offset (nsec) to add */
    EC_T_DWORD      dwTimeout;                      /*< in  timeout for proceeding add offset */
}EC_PACKED EC_T_DC_SYSTIME_ADD_OFFSET_DESC;

typedef struct _EC_T_DC_SYNCSO_CB_PARM
{
    EC_T_VOID*              pvContext;          /* in       Calling environment context */
    EC_T_DWORD              dwTimeStamp;        /* in       Pre-Send Timestamp */
    EC_T_DWORD              dwPostTimeStamp;    /* in       Post-Send Timestamp */
    EC_T_DWORD              dwBusTimeLo;        /* in       Bus Time Lower 32 Bit */
    EC_T_DWORD              dwBusTimeHi;        /* in       Bus Time Upper 32 Bit */
    EC_T_DWORD              dwTimeStampResult;  /* in       Result of stamp */

    EC_T_DWORD	        dwSyncPeriodLength; /* in       Time between two consecutive SYNC0 signals of clock master slave in nsec. */
        
    EC_T_DWORD              dwStartSyncTimeLo;  /* out      Sync Start Time Lower 32 Bit */
    EC_T_DWORD              dwStartSyncTimeHi;  /* out      Sync Start Time Upper 32 Bit */

}EC_PACKED  EC_T_DC_SYNCSO_CB_PARM, *EC_PT_DC_SYNCSO_CB_PARM;

typedef EC_T_DWORD (*EC_T_PFSYNCSO_CB)(EC_T_DC_SYNCSO_CB_PARM* pParm);
       
typedef struct _EC_T_DC_SYNCSO_REGDESC
{
    EC_T_DC_SYNCSO_CB_PARM*     pCallbackParm;      /*< in  Callback Parameter */
    EC_T_PFSYNCSO_CB            pfnCallback;        /*< in  Callback Function */
}EC_PACKED  EC_T_DC_SYNCSO_REGDESC, *EC_PT_DC_SYNCSO_REGDESC;
  
 
typedef struct _EC_T_DC_CONFIGURE
{
    EC_T_DWORD                  dwClntId;           /*< in  client ID */
    EC_T_DWORD                  dwTimeout;          /*< in  DC Timeout value in msec */
    EC_T_DWORD                  dwDevLimit;         /*< in  deviation limit value*/
    EC_T_DWORD                  dwSettleTime;       /*< in  config values for settle time*/
    EC_T_DWORD	                dwTotalBurstLength;
    EC_T_DWORD	                dwBurstBulk;
    EC_T_DWORD                  dwReserved[5];      /*< in/out 5 DWORD reserved*/
}EC_PACKED  EC_T_DC_CONFIGURE, *EC_PT_DC_CONFIGURE;

/* ecatGetSlaveInfo */
typedef struct _EC_T_GET_SLAVE_INFO
{
    EC_T_DWORD                  dwScanBusStatus;    /* 0x00 */  /*< Status during last Bus Scan */
    
    EC_T_DWORD	                dwVendorId;         /* 0x01 */  /*< Vendor Identification */
    EC_T_DWORD	                dwProductCode;      /* 0x02 */  /*< Product Code */
    EC_T_DWORD	                dwRevisionNumber;   /* 0x03 */  /*< Revision Number */
    EC_T_DWORD	                dwSerialNumber;     /* 0x04 */  /*< Serial Number */
    
    EC_T_WORD                   wPortState;         /* 0x05 */  /*< out port link state (SB Instance)*/
    EC_T_WORD                   wReserved;                      /*< Res */

    EC_T_BOOL                   bDcSupport;         /* 0x06 */  /*< out slave does support DC*/
    EC_T_BOOL                   bDc64Support;       /* 0x07 */  /*< out slave does support 64Bit DC*/
    
    EC_T_WORD                   wAliasAddress;      /* 0x08 */  /*< out slave alias address*/
    EC_T_WORD                   wPhysAddress;                   /*< out slave physical address*/
    
    EC_T_DWORD                  dwPdOffsIn;         /* 0x09 */  /*< out process data offset of Input Data (in Bits)*/
    EC_T_DWORD                  dwPdSizeIn;         /* 0x0A */  /*< out process data size of Input Data (in Bits)*/
    EC_T_DWORD                  dwPdOffsOut;        /* 0x0B */  /*< out process data offset of Output Data (in Bits)*/
    EC_T_DWORD                  dwPdSizeOut;        /* 0x0C */  /*< out process data size of Output Data*/
    EC_T_DWORD                  dwPdOffsIn2;        /* 0x0D */  /*< out process data offset of Input data (in Bits)*/
    EC_T_DWORD                  dwPdSizeIn2;        /* 0x0E */  /*< out process data size of Input Data (in Bits)*/
    EC_T_DWORD                  dwPdOffsOut2;       /* 0x0F */  /*< out process data offset of Output Data (in Bits)*/
    EC_T_DWORD                  dwPdSizeOut2;       /* 0x10 */  /*< out process data size of Output Data*/
    EC_T_DWORD                  dwPdOffsIn3;        /* 0x11 */  /*< out process data offset of Input Data (in Bits)*/
    EC_T_DWORD                  dwPdSizeIn3;        /* 0x12 */  /*< out process data size of Input Data (in Bits)*/
    EC_T_DWORD                  dwPdOffsOut3;       /* 0x13 */  /*< out process data offset of Output Data (in Bits)*/
    EC_T_DWORD                  dwPdSizeOut3;       /* 0x14 */  /*< out process data size of Output Data*/
    EC_T_DWORD                  dwPdOffsIn4;        /* 0x15 */  /*< out process data offset of Input Data (in Bits)*/
    EC_T_DWORD                  dwPdSizeIn4;        /* 0x16 */  /*< out process data size of Input Data (in Bits)*/
    EC_T_DWORD                  dwPdOffsOut4;       /* 0x17 */  /*< out process data offset of Output Data (in Bits)*/
    EC_T_DWORD                  dwPdSizeOut4;       /* 0x18 */  /*< out process data size of Output Data*/
    
    EC_T_WORD                   wCfgPhyAddress;     /* 0x19 */  /*< out slave configured physical address*/
    EC_T_WORD                   wReserved2;                     /*< reserved */
    
    EC_T_CHAR                   abyDeviceName[ECAT_DEVICE_NAMESIZE];    
                                                    /* 0x1A */  /*< out slave name of configuration*/
    EC_T_BOOL                   bIsMailboxSlave;    /* 0x2E */  /*< out whether slave support mailboxes*/
    EC_T_DWORD                  dwMbxOutSize;       /* 0x2F */  /*< out mailbox 1 output size*/
    EC_T_DWORD                  dwMbxInSize;        /* 0x30 */  /*< out mailbox 1 input size*/
    EC_T_DWORD                  dwMbxOutSize2;      /* 0x31 */  /*< out mailbox 2 output size*/
    EC_T_DWORD                  dwMbxInSize2;       /* 0x32 */  /*< out mailbox 2 input size*/

    EC_T_DWORD                  dwErrorCode;        /* 0x33 */  /*< out last return code*/
    EC_T_DWORD                  dwSBErrorCode;      /* 0x34 */  /*< out last return value from SB*/

    EC_T_BYTE                   byPortDescriptor;   /* 0x35 */  /*< out Port Descriptor */
    EC_T_BYTE                   byESCType;                      /*< out ESC Node Type */
    EC_T_WORD                   wReserved3;                     /*< reserved */
    
    EC_T_WORD                   wAlStatusValue;     /* 0x36 */  /*< out AL Status Register Value */
    EC_T_WORD                   wAlStatusCode;                  /*< out AL Status Code */

    EC_T_BOOL                   bIsOptional;        /* 0x37 */  /*< out slave is in an optional hot connect group */
    EC_T_BOOL                   bIsPresent;         /* 0x38 */  /*< out slave is currently present on bus */

    EC_T_WORD                   wNumProcessVarsInp; /* 0x39 */  /*< out number of output process data variables*/
    EC_T_WORD                   wNumProcessVarsOutp;            /*< out number of input process data variables */

    EC_T_DWORD                  dwReserved[25];     /* 0x3A */  /*< out Reserved*/
                                               
} EC_PACKED  EC_T_GET_SLAVE_INFO, *EC_PT_GET_SLAVE_INFO;
                                                    /* Size = 0x53 * DWORD */  

/* emGetSlaveInpVarInfo / emGetSlaveOutpVarInfo*/
typedef struct _EC_T_PROCESS_VAR_INFO
{
    EC_T_CHAR                   szName[MAX_PROCESS_VAR_NAME_LEN];   /**< [out] Name of the found process variable */
    EC_T_WORD                   wDataType;                          /**< [out] Data type of the found process variable */
    EC_T_WORD                   wFixedAddr;                         /**< [out] Fixed station address of the slave that is owner of this variable */
    EC_T_INT                    nBitSize;                           /**< [out] Size in bit of the found process variable */
    EC_T_INT                    nBitOffs;                           /**< [out] Bit offset in the process data image */
    EC_T_BOOL                   bIsInputData;                       /**< [out] Determines whether the found process variable is an input variable or an output variable */
} EC_PACKED EC_T_PROCESS_VAR_INFO, *EC_PT_PROCESS_VAR_INFO;


/* EtherCAT CoE OD list type values */
typedef enum _EC_T_COE_ODLIST_TYPE
{
    eODListType_Lengths     = 0,                    /*< lengths of each list type */
    eODListType_ALL         = 1,                    /*< all objects */
    eODListType_RxPdoMap    = 2,                    /*< only pdo mappable objects */
    eODListType_TxPdoMap    = 3,                    /*< only pdo mappable objects that can be changed */
    eODListType_StoredFRepl = 4,                    /*< only stored for a device replacement objects */
    eODListType_StartupParm = 5,                    /*< only startup parameter objects */

    /* Borland C++ datatype alignment correction */
    eODListType_BCppDummy   = 0xFFFFFFFF
} EC_T_COE_ODLIST_TYPE;
#define CoeOdListTypeText(EType)                                \
    ((EType)==eODListType_Lengths?"Lengths":                    \
     ((EType)==eODListType_ALL?"All":                           \
      ((EType)==eODListType_RxPdoMap?"RxPDO":                   \
       ((EType)==eODListType_TxPdoMap?"TxPDO":                  \
        ((EType)==eODListType_StoredFRepl?"Device Replacement": \
         ((EType)==eODListType_StartupParm?"Startup Param":     \
         "Unknown"                                              \
    ))))))


/* EtherCAT CoE OD list */
typedef struct _EC_T_COE_ODLIST
{
    EC_T_COE_ODLIST_TYPE    eOdListType;            /*< type of OD list */
    EC_T_WORD               wLen;                   /*< length of the OD list */
    EC_T_WORD*              pwOdList;               /*< OD list */
}EC_PACKED EC_T_COE_ODLIST;                                  
                                                    
/* EtherCAT CoE Object description */               
typedef struct _EC_T_COE_OBDESC                     
{                                                   
    EC_T_WORD   wObIndex;                           /*< object index */
    EC_T_WORD   wDataType;                          /*< data type */
    EC_T_BYTE   byObjCode;                          /*< object code defined in DS 301 (Table 37) */
    EC_T_BYTE   byObjCategory;                      /*< category: 0=optional, 1=mandatory */
    EC_T_BYTE   byMaxNumSubIndex;                   /*< maximum number of sub-indexes */
    EC_T_WORD   wObNameLen;                         /*< length of the object name */
    EC_T_CHAR*  pchObName;                          /*< object name (not a NULL terminated string!) */
}EC_PACKED EC_T_COE_OBDESC;

/* EtherCAT CoE Object Entry description */
typedef struct _EC_T_COE_ENTRYDESC
{
    EC_T_WORD   wObIndex;                           /*< object index */
    EC_T_BYTE   byObSubIndex;                       /*< object sub-index */
    EC_T_BYTE   byValueInfo;                        /*< value information, determines which of the following elements are valid */
    EC_T_WORD   wDataType;                          /*< data type */
    EC_T_WORD   wBitLen;                            /*< bit length of the object */
    EC_T_BYTE   byObAccess;                         /*< object access */
    EC_T_BOOL   bRxPdoMapping;                      /*< EC_TRUE if it is PDO mappable */
    EC_T_BOOL   bTxPdoMapping;                      /*< EC_TRUE if PDO mappable object can be changed */
    EC_T_BOOL   bObCanBeUsedForBackup;              /*< Parameter may be back upped */
    EC_T_BOOL   bObCanBeUsedForSettings;            /*< Parameter may be used for Settings */
    EC_T_WORD   wUnitType;                          /*< unit type of the object, only valid if EC_COE_ENTRY_UnitType bit is set in byValueInfo */
    EC_T_WORD   wDataLen;                           /*< length of the data */
    EC_T_BYTE*  pbyData;                            /*< rest of the entry description, content see below */
    /*EC_T_BYTE   pbyDefaultValue[];                 *< default value, only valid if EC_COE_ENTRY_DefaultValue bit is set in byValueInfo */
    /*EC_T_BYTE   pbyMinValue[];                     *< minimum value, only valid if EC_COE_ENTRY_MinValue bit is set in byValueInfo */
    /*EC_T_BYTE   pbyMaxValue[];                     *< maximum value, only valid if EC_COE_ENTRY_MaxValue bit is set in byValueInfo */
    /*EC_T_BYTE   pbyDescription[];                  *< object description */
}EC_PACKED EC_T_COE_ENTRYDESC;                               
                                                    
/* EtherCAT CoE emergency request */                
#define EC_COE_EMERGENCY_DATASIZE 5                 
typedef struct _EC_T_COE_EMERGENCY                  
{                                                   
    EC_T_WORD   wErrorCode;                         /*< error code, see specification for details */
    EC_T_BYTE   byErrorRegister;                    /*< error register */
    EC_T_BYTE   abyData[EC_COE_EMERGENCY_DATASIZE]; /*< error data */
    EC_T_WORD   wStationAddress;                    /*< which slave caused the emergency request */
}EC_PACKED EC_T_COE_EMERGENCY;                               

#ifdef INCLUDE_SOE_SUPPORT
/* EtherCAT SoE emergency request */                
#define EC_SOE_EMERGENCY_DATASIZE 5                 
typedef struct _EC_T_SOE_EMERGENCY                  
{                                                   
    EC_T_WORD   wHeader;                            /*< SoE Header */
    EC_T_BYTE   abyData[EC_SOE_EMERGENCY_DATASIZE]; /*< error data */
    EC_T_WORD   wStationAddress;                    /*< which slave caused the emergency request */
}EC_PACKED EC_T_SOE_EMERGENCY;                               

#define EC_SOE_NOTIFICATION_DATASIZE 5                 
typedef struct _EC_T_SOE_NOTIFICATION                  
{                                                   
    EC_T_WORD   wHeader;                                /*< SoE Header */
    EC_T_WORD   wIdn;                                   /*< IDN number of the appendent call */
    EC_T_BYTE   abyData[EC_SOE_NOTIFICATION_DATASIZE];  /*< error data */
    EC_T_WORD   wStationAddress;                        /*< which slave caused the emergency request */
}EC_PACKED EC_T_SOE_NOTIFICATION;                               

#endif
                                                    
/* EtherCAT mailbox data */                         
typedef union _EC_T_MBX_DATA                        
{                                                   
    EC_T_COE_ODLIST         CoE_ODList;                 /*< CoE object dictionary list */
    EC_T_COE_OBDESC         CoE_ObDesc;                 /*< CoE object description */
    EC_T_COE_ENTRYDESC      CoE_EntryDesc;              /*< CoE object entry description */
    EC_T_COE_EMERGENCY      CoE_Emergency;              /*< CoE emergency request */
#ifdef INCLUDE_SOE_SUPPORT
    EC_T_SOE_EMERGENCY      SoE_Emergency;              /*< SoE emergency request */
    EC_T_SOE_NOTIFICATION   SoE_Notification;           /*< SoE notification request */
#endif
}EC_PACKED EC_T_MBX_DATA;

/* EtherCAT mailbox type values */
typedef enum _EC_T_MBXTFER_TYPE
{
    eMbxTferType_COE_SDO_DOWNLOAD   = 0,              /*< CoE SDO download */
    eMbxTferType_COE_SDO_UPLOAD     = 1,              /*< CoE SDO upload */
    eMbxTferType_COE_GETODLIST      = 2,              /*< CoE Get object dictionary list */
    eMbxTferType_COE_GETOBDESC      = 3,              /*< CoE Get object description */
    eMbxTferType_COE_GETENTRYDESC   = 4,              /*< CoE Get object entry description */
    eMbxTferType_COE_EMERGENCY      = 5,              /*< CoE emergency request */
#ifdef INCLUDE_COE_PDO_SUPPORT      
    eMbxTferType_COE_RX_PDO         = 6,              /*< CoE RxPDO */
#endif                              
    eMbxTferType_FOE_FILE_UPLOAD    = 7,             /*<  FOE upload */
    eMbxTferType_FOE_FILE_DOWNLOAD  = 8,             /*<  FOE Download */
#ifdef INCLUDE_SOE_SUPPORT
    eMbxTferType_SOE_READREQUEST    = 9,             /*<  SOE types */
    eMbxTferType_SOE_READRESPONSE   = 10,
    eMbxTferType_SOE_WRITEREQUEST   = 11,
    eMbxTferType_SOE_WRITERESPONSE  = 12,
    eMbxTferType_SOE_NOTIFICATION   = 13,
    eMbxTferType_SOE_EMERGENCY      = 14,
#endif
    /* Borland C++ datatype alignment correction */
    eMbxTferType_BCppDummy          = 0xFFFFFFFF
} EC_T_MBXTFER_TYPE;

/* EtherCAT mailbox transfer descriptor */
typedef struct _EC_T_MBXTFER_DESC
{
    EC_T_DWORD          dwMaxDataLen;               /*< maximum number of bytes to be transferred */
    EC_T_BYTE*          pbyMbxTferDescData;         /*< Data buffer to hold transfer data */
}EC_PACKED EC_T_MBXTFER_DESC;
                                                    
/* EtherCAT mailbox status values */                
typedef enum _EC_T_MBXTFER_STATUS                   
{
    eMbxTferStatus_Idle         = 0,                /*< Mailbox transfer object currently idle, no transfer running */
    eMbxTferStatus_Pend         = 1,                /*< Mailbox transfer object currently idle, no transfer running */
    eMbxTferStatus_TferDone     = 2,                /*< transfer successfully executed */
    eMbxTferStatus_TferReqError = 3,                /*< transfer request error occurred */

    /* Borland C++ datatype alignment correction */
    eMbxTferStatus_BCppDummy    = 0xFFFFFFFF
} EC_T_MBXTFER_STATUS;                              
                                                    
/* EtherCAT mailbox transfer object */              
typedef struct _EC_T_MBXTFER                        
{                                                   
    EC_T_DWORD          dwClntId;                   /*< client ID */
    EC_T_MBXTFER_DESC   MbxTferDesc;                /*< mailbox transfer descriptor */
    EC_T_MBXTFER_TYPE   eMbxTferType;               /*< mailbox transfer type */
    EC_T_DWORD          dwDataLen;                  /*< number of bytes to be transferred */
    EC_T_BYTE*          pbyMbxTferData;             /*< pointer to mailbox data */
    EC_T_MBXTFER_STATUS eTferStatus;                /*< current transfer status */
    EC_T_DWORD          dwErrorCode;                /*< transfer error code */
    EC_T_DWORD          dwTferId;                   /*< unique transfer ID */
    EC_T_MBX_DATA       MbxData;                    /*< mailbox data */
}EC_PACKED EC_T_MBXTFER;


/* Supported EtherCAT commands for the ecatTferSingleRawCmd() function */
typedef enum _EC_T_RAWCMD
{
    eRawCmd_APRD    = EC_CMD_TYPE_APRD,     /* auto-increment physical read */
    eRawCmd_APWR    = EC_CMD_TYPE_APWR,     /* auto-increment physical write */
    eRawCmd_APRW    = EC_CMD_TYPE_APRW,     /* auto-increment physical read/write */
    eRawCmd_BRD     = EC_CMD_TYPE_BRD,      /* broadcast (wire-or) read */
    eRawCmd_BWR     = EC_CMD_TYPE_BWR,      /* broadcast write */
    eRawCmd_BRW     = EC_CMD_TYPE_BRW,      /* broadcast read/write */
    eRawCmd_LRD     = EC_CMD_TYPE_LRD,      /* logical read */
    eRawCmd_LWR     = EC_CMD_TYPE_LWR,      /* logical write */
    eRawCmd_LRW     = EC_CMD_TYPE_LRW,      /* logical read/write */
    eRawCmd_ARMW    = EC_CMD_TYPE_ARMW,     /* auto-increment physical read, multiple write */
    eRawCmd_FPRD    = EC_CMD_TYPE_FPRD,     /* fixed address physical read */
    eRawCmd_FPWR    = EC_CMD_TYPE_FPWR,     /* fixed address physical write */
    eRawCmd_FPRW    = EC_CMD_TYPE_FPRW,     /* fixed address physical read/write */

    /* Borland C++ datatype alignment correction */
    eRawCmd_BCppDummy   = 0xFFFFFFFF
} EC_T_RAWCMD;

/* Descriptor for EC_IOCTL_GET_PDMEMORYSIZE call */
typedef struct _EC_T_MEMREQ_DESC
{
    EC_T_DWORD      dwPDOutSize;                    /* Size of the output process data image */
    EC_T_DWORD      dwPDInSize;                     /* Size of the input  process data image */
}EC_PACKED EC_T_MEMREQ_DESC, *EC_PT_MEMREQ_DESC;

/* Descriptor for EC_IOCTL_UPDATE_ALL_SLAVE_STATE call */
typedef struct _EC_T_SLAVE_STATE_DESC
{
    EC_T_DWORD  dwNumSlaves;        /* [in]	 size of the awCurrDevState[] and awReqDevState[] arrays */
    EC_T_BOOL   bUseFixedAddr;      /* [in]	 EC_TRUE if fixed station address shall be used, EC_FALSE if auto-increment address shall be used */
    EC_T_DWORD  dwNumRespSlaves;    /* [out] number of slaves responding */
    EC_T_DWORD* adwSlaveId;         /* [out] slave id */
    EC_T_BOOL*  abSlaveResponded;   /* [out] EC_TRUE if slave responded */
    EC_T_WORD*  awCurrDevState;     /* [out] current slave device state */
    EC_T_WORD*  awReqDevState;      /* [out] requested slave device state */
}EC_PACKED EC_T_SLAVE_STATE_DESC;

/* Descriptor for EC_IOCTL_HC_CONFIGURETIMEOUTS call */
typedef struct _EC_T_HC_CONFIGURETIMEOUTS_DESC
{
    EC_T_DWORD  dwDetectionTimeout;         /* [in] Timeout for Group detection */
    EC_T_DWORD  dwTopologyChangeTimeout;    /* [in] Topology Change Timeout (usually larger than dwDetectionTimeout) */
    EC_T_DWORD  dwRsvd[2];
}EC_PACKED EC_T_HC_CONFIGURETIMEOUTS_DESC;

/* EtherCAT user controlled execution job */
typedef enum _EC_T_USER_JOB
{
    eUsrJob_ProcessAllRxFrames              = 1,        /*< polling mode: receive frames and process all received data */
    eUsrJob_SendAllCycFrames                = 2,        /*< send all cyclic frames */
    eUsrJob_RunMcSm                         = 3,        /*< run the master control state machine */
    eUsrJob_MasterTimer                     = 4,        /*< run master timer routine (generic management) */
    eUsrJob_FlushQueuedCmds                 = 5,        /*< obsolete */
    eUsrJob_SendAcycFrames                  = 6,        /*< send acyclic frames (number of frames can be limited by master parameter) */
    eUsrJob_SendCycFramesByTaskId           = 7,        /*< send cyclic frames related to a specific task id (TaskId entry in the XML file) */
                                            
    eUsrJob_ProcessAllCycFrames             = 10,       /*< polling mode: process all cyclic frames  */
    eUsrJob_ProcessAllAcycFrames            = 11,       /*< polling mode: process all acyclic frames */

    eUsrJob_StampSendAllCycFrames           = 22,       /*< send all cyclic frames (stamped) */
    eUsrJob_StampSendCycFramesByTaskId      = 27,        /*< send cyclic frames related to a specific task id (TaskId entry in the XML file) */

    /* Borland C++ datatype alignment correction */
    eUsrJob_BCppDummy                       = 0xFFFFFFFF
} EC_T_USER_JOB;


typedef EC_T_VOID (*EC_PF_CYCFRAME_RECV)(EC_T_VOID*);
typedef struct _EC_T_CYCFRAME_RX_CBDESC
{
    EC_T_VOID*                  pCallbackParm;      /*< in  Callback Parameter */
    EC_PF_CYCFRAME_RECV         pfnCallback;        /*< in  Callback Function */
}EC_PACKED  EC_T_CYCFRAME_RX_CBDESC;


/*-PACK SETTINGS-------------------------------------------------------------*/
#if defined  __GNUC__   /* GNU */

#undef  EC_PACKED

#if (defined __RCX__)
#pragma pack ()
#endif

#elif defined _MSC_VER  /* MICROSOFT */

#undef  EC_PACKED
#pragma pack (pop)

#elif defined __MET__

#undef EC_PACKED
/*#pragma pack (pop)*/
#pragma Pop_align_members()

#else
#endif

/*-FORWARD DECLARATIONS------------------------------------------------------*/

#ifdef __cplusplus
extern "C"
{
#endif

ATECAT_API EC_T_CHAR*  EcErrorText(                 EC_T_DWORD              dwError         );
const EC_T_CHAR*       EcStateString(               EC_T_STATE              eMasterState    );
                                                    
ATECAT_API EC_T_CHAR*  ecatGetText(                 EC_T_WORD               wTextId         );
ATECAT_API EC_T_DWORD  ecatInitMaster(              EC_T_INITMASTERPARMS*   pParms          );
ATECAT_API EC_T_DWORD  ecatDeinitMaster(            EC_T_VOID                               );
                                                    
ATECAT_API EC_T_DWORD  ecatStart(                   EC_T_DWORD              dwTimeout       );
ATECAT_API EC_T_DWORD  ecatStop(                    EC_T_DWORD              dwTimeout       );
                                                    
ATECAT_API EC_T_DWORD  ecatIoControl(               EC_T_DWORD              dwCode, 
                                                    EC_T_IOCTLPARMS*        pParms          );
                                                    
ATECAT_API EC_T_DWORD  ecatGetSlaveId(              EC_T_WORD               wStationAddress );

ATECAT_API EC_T_DWORD  ecatGetSlaveFixedAddr(       EC_T_DWORD              dwSlaveId,              
                                                    EC_T_WORD*              pwFixedAddr     );

ATECAT_API EC_T_DWORD  ecatGetSlaveIdAtPosition(    EC_T_WORD               wAutoIncAddress );
ATECAT_API EC_T_BOOL   ecatGetSlaveProp(            EC_T_DWORD              dwSlaveId, 
                                                    EC_T_SLAVE_PROP*        pSlaveProp      );
ATECAT_API EC_T_DWORD  ecatGetSlaveState(           EC_T_DWORD              dwSlaveId, 
                                                    EC_T_WORD*              pwCurrDevState, 
                                                    EC_T_WORD*              pwReqDevState, 
                                                    EC_T_DWORD              dwTimeout       );
ATECAT_API EC_T_DWORD  ecatSetSlaveState(           EC_T_DWORD              dwSlaveId, 
                                                    EC_T_WORD               wNewReqDevState, 
                                                    EC_T_DWORD              dwTimeout       );
ATECAT_API EC_T_DWORD  ecatTferSingleRawCmd(        EC_T_DWORD              dwSlaveId, 
                                                    EC_T_BYTE               byCmd, 
                                                    EC_T_DWORD              dwMemoryAddress, 
                                                    EC_T_VOID*              pvData, 
                                                    EC_T_WORD               wLen, 
                                                    EC_T_DWORD              dwTimeout       );
ATECAT_API EC_T_DWORD  ecatQueueRawCmd(             EC_T_DWORD              dwSlaveId,
                                                    EC_T_WORD               wInvokeId,
                                                    EC_T_BYTE               byCmd,
                                                    EC_T_DWORD              dwMemoryAddress,
                                                    EC_T_VOID*              pvData,
                                                    EC_T_WORD               wLen            );
ATECAT_API EC_T_DWORD  ecatClntQueueRawCmd(         EC_T_DWORD              dwClntId,
                                                    EC_T_DWORD              dwSlaveId,
                                                    EC_T_WORD               wInvokeId,
                                                    EC_T_BYTE               byCmd,
                                                    EC_T_DWORD              dwMemoryAddress,
                                                    EC_T_VOID*              pvData,
                                                    EC_T_WORD               wLen            );
ATECAT_API EC_T_DWORD  ecatGetNumConfiguredSlaves(  EC_T_VOID                               );


ATECAT_API EC_T_MBXTFER* ecatMbxTferCreate(         EC_T_MBXTFER_DESC*      pMbxTferDesc    );
ATECAT_API EC_T_VOID     ecatMbxTferDelete(         EC_T_MBXTFER*           pMbxTfer        );
                                                    
ATECAT_API EC_T_DWORD  ecatCoeSdoDownloadReq(       EC_T_MBXTFER*           pMbxTfer, 
                                                    EC_T_DWORD              dwSlaveId, 
                                                    EC_T_WORD               wObIndex, 
                                                    EC_T_BYTE               byObSubIndex, 
                                                    EC_T_DWORD              dwTimeout,
                                                    EC_T_DWORD              dwFlags         );
ATECAT_API EC_T_DWORD   ecatCoeSdoDownload(         EC_T_DWORD              dwSlaveId
                                                   ,EC_T_WORD               wObIndex
                                                   ,EC_T_BYTE               byObSubIndex
                                                   ,EC_T_BYTE*              pbyData
                                                   ,EC_T_DWORD              dwDataLen
                                                   ,EC_T_DWORD              dwTimeout
                                                   ,EC_T_DWORD              dwFlags         );
ATECAT_API EC_T_DWORD  ecatCoeSdoUploadReq(         EC_T_MBXTFER*           pMbxTfer, 
                                                    EC_T_DWORD              dwSlaveId, 
                                                    EC_T_WORD               wObIndex, 
                                                    EC_T_BYTE               byObSubIndex, 
                                                    EC_T_DWORD              dwTimeout,
                                                    EC_T_DWORD              dwFlags         );
ATECAT_API EC_T_DWORD  ecatCoeSdoUpload(            EC_T_DWORD              dwSlaveId
                                                   ,EC_T_WORD               wObIndex
                                                   ,EC_T_BYTE               byObSubIndex
                                                   ,EC_T_BYTE*              pbyData
                                                   ,EC_T_DWORD              dwDataLen
                                                   ,EC_T_DWORD*             pdwOutDataLen
                                                   ,EC_T_DWORD              dwTimeout
                                                   ,EC_T_DWORD              dwFlags         );
ATECAT_API EC_T_DWORD  ecatCoeGetODList(            EC_T_MBXTFER*           pMbxTfer, 
                                                    EC_T_DWORD              dwSlaveId, 
                                                    EC_T_COE_ODLIST_TYPE    eListType, 
                                                    EC_T_DWORD              dwTimeout       );
ATECAT_API EC_T_DWORD  ecatCoeGetObjectDesc(        EC_T_MBXTFER*           pMbxTfer, 
                                                    EC_T_DWORD              dwSlaveId, 
                                                    EC_T_WORD               wObIndex, 
                                                    EC_T_DWORD              dwTimeout       );
ATECAT_API EC_T_DWORD  ecatCoeGetEntryDesc(         EC_T_MBXTFER*           pMbxTfer, 
                                                    EC_T_DWORD              dwSlaveId, 
                                                    EC_T_WORD               wObIndex, 
                                                    EC_T_BYTE               byObSubIndex, 
                                                    EC_T_BYTE               byValueInfo,
                                                    EC_T_DWORD              dwTimeout       );

#ifdef INCLUDE_COE_PDO_SUPPORT
ATECAT_API EC_T_DWORD ecatCoeRxPdoTfer(             EC_T_MBXTFER*           pMbxTfer,
                                                    EC_T_DWORD              dwSlaveId,
                                                    EC_T_DWORD              dwNumber,
                                                    EC_T_DWORD              dwTimeout       );
#endif

ATECAT_API EC_T_DWORD  ecatFoeFileUpload(           EC_T_DWORD              dwSlaveId,
                                                    EC_T_CHAR*              szFileName,
                                                    EC_T_DWORD              dwFileNameLen,
                                                    EC_T_BYTE*              pbyData,
                                                    EC_T_DWORD              dwDataLen,
                                                    EC_T_DWORD*             pdwOutDataLen,
                                                    EC_T_DWORD              dwPassWd,
                                                    EC_T_DWORD              dwTimeout       );

ATECAT_API EC_T_DWORD  ecatFoeFileDownload(         EC_T_DWORD              dwSlaveId,
                                                    EC_T_CHAR*              szFileName,
                                                    EC_T_DWORD              dwFileNameLen,
                                                    EC_T_BYTE*              pbyData,
                                                    EC_T_DWORD              dwDataLen,
                                                    EC_T_DWORD              dwPassWd,
                                                    EC_T_DWORD              dwTimeout       );

ATECAT_API EC_T_DWORD  ecatFoeUploadReq(            EC_T_MBXTFER*           pMbxTfer,
                                                    EC_T_DWORD              dwSlaveId,
                                                    EC_T_CHAR*              szFileName,
                                                    EC_T_DWORD              dwFileNameLen,
                                                    EC_T_DWORD              dwPassWd,
                                                    EC_T_DWORD              dwTimeout       );

ATECAT_API EC_T_DWORD  ecatFoeDownloadReq(          EC_T_MBXTFER*           pMbxTfer,
                                                    EC_T_DWORD              dwSlaveId,
                                                    EC_T_CHAR*              szFileName,
                                                    EC_T_DWORD              dwFileNameLen,
                                                    EC_T_DWORD              dwPassWd,
                                                    EC_T_DWORD              dwTimeout       );

#ifdef INCLUDE_SOE_SUPPORT
ATECAT_API  EC_T_DWORD   ecatSoeWrite(              EC_T_DWORD              dwSlaveId,
                                                    EC_T_BYTE               byDriveNo,
                                                    EC_T_BYTE               byElementFlags,
                                                    EC_T_WORD               wIDN,
                                                    EC_T_BYTE*              pbyData,
                                                    EC_T_DWORD              dwDataLen,
                                                    EC_T_DWORD              dwTimeout        );
ATECAT_API  EC_T_DWORD ecatSoeRead(                 EC_T_DWORD              dwSlaveId,
                                                    EC_T_BYTE               byDriveNo,
                                                    EC_T_BYTE               byElementFlags,
                                                    EC_T_WORD               wIDN, 
                                                    EC_T_BYTE*              pbyData,
                                                    EC_T_DWORD              dwDataLen,  
                                                    EC_T_DWORD*             pdwOutDataLen,
                                                    EC_T_DWORD              dwTimeout        );
ATECAT_API  EC_T_DWORD   ecatSoeAbortProcCmd(       EC_T_DWORD              dwSlaveId,
                                                    EC_T_BYTE               byDriveNo,
                                                    EC_T_BYTE               byElementFlags,
                                                    EC_T_WORD               wIDN,
                                                    EC_T_DWORD              dwTimeout        );
#endif                                              
                                                    
ATECAT_API EC_T_DWORD  ecatEoeRegisterEndpoint(     EC_T_CHAR*              szEoEDrvIdent,
                                                    EC_T_VOID*              pLinkDrvDesc    ); 

ATECAT_API EC_T_DWORD  ecatConfigureMaster(         EC_T_CNF_TYPE           eCnfType,
                                                    EC_T_PBYTE              pbyCnfData, 
                                                    EC_T_DWORD              dwCnfDataLen    );

ATECAT_API EC_T_DWORD  ecatSetMasterState(          EC_T_DWORD              dwTimeout,
                                                    EC_T_STATE              eReqState       );
ATECAT_API EC_T_STATE  ecatGetMasterState(          EC_T_VOID                               );

ATECAT_API EC_T_DWORD  ecatExecJob(                 EC_T_USER_JOB           eUserJob,
                                                    EC_T_PVOID              pvParam         );

ATECAT_API EC_T_DWORD  ecatGetProcessData(          EC_T_BOOL               bOutputData,
                                                    EC_T_DWORD              dwOffset,
                                                    EC_T_BYTE*              pbyData,
                                                    EC_T_DWORD              dwLength,
                                                    EC_T_DWORD              dwTimeout       );
ATECAT_API EC_T_DWORD  ecatSetProcessData(          EC_T_BOOL               bOutputData,
                                                    EC_T_DWORD              dwOffset,
                                                    EC_T_BYTE*              pbyData,
                                                    EC_T_DWORD              dwLength,
                                                    EC_T_DWORD              dwTimeout       );
ATECAT_API EC_T_DWORD  ecatGetNumConnectedSlaves(   EC_T_VOID                               );

ATECAT_API EC_T_DWORD  ecatReadSlaveEEPRom(         EC_T_BOOL               bFixedAddressing,
                                                    EC_T_WORD               wSlaveAddress,
                                                    EC_T_WORD               wEEPRomStartOffset,
                                                    EC_T_WORD*              pwReadData,
                                                    EC_T_DWORD              dwReadLen,
                                                    EC_T_DWORD*             pdwNumOutData,
                                                    EC_T_DWORD              dwTimeout       );
ATECAT_API EC_T_DWORD ecatWriteSlaveEEPRom(         EC_T_BOOL               bFixedAddressing,
                                                    EC_T_WORD               wSlaveAddress,
                                                    EC_T_WORD               wEEPRomStartOffset,
                                                    EC_T_WORD*              pwWriteData,
                                                    EC_T_DWORD              dwWriteLen,
                                                    EC_T_DWORD              dwTimeout       );
ATECAT_API EC_T_DWORD ecatReloadSlaveEEPRom(        EC_T_BOOL               bFixedAddressing,
                                                    EC_T_WORD               wSlaveAddress,
                                                    EC_T_DWORD              dwTimeout       );
ATECAT_API EC_T_DWORD ecatResetSlaveController(     EC_T_BOOL               bFixedAddressing,
                                                    EC_T_WORD               wSlaveAddress,
                                                    EC_T_DWORD              dwTimeout       );
ATECAT_API EC_T_DWORD ecatAssignSlaveEEPRom(        EC_T_BOOL               bFixedAddressing,
                                                    EC_T_WORD               wSlaveAddress,
                                                    EC_T_BOOL               bSlavePDIAccessEnable,
                                                    EC_T_BOOL               bForceAssign,
                                                    EC_T_DWORD              dwTimeout       );
ATECAT_API EC_T_DWORD ecatActiveSlaveEEPRom(        EC_T_BOOL               bFixedAddressing,
                                                    EC_T_WORD               wSlaveAddress,
                                                    EC_T_BOOL*              pbSlavePDIAccessActive,
                                                    EC_T_DWORD              dwTimeout       );

ATECAT_API EC_T_DWORD ecatHCAcceptTopoChange(       EC_T_VOID                               );
ATECAT_API EC_T_DWORD ecatHCGetNumGroupMembers(     EC_T_DWORD              dwGroupIndex    );
ATECAT_API EC_T_DWORD ecatHCGetSlaveIdsOfGroup(     EC_T_DWORD              dwGroupIndex,
                                                    EC_T_DWORD*             adwSlaveId,
                                                    EC_T_DWORD              dwMaxNumSlaveIds );
ATECAT_API EC_T_DWORD ecatSetSlavePortState(        EC_T_DWORD              dwSlaveID,
                                                    EC_T_WORD               wPort,
                                                    EC_T_BOOL               bClose,
                                                    EC_T_BOOL               bForce,
                                                    EC_T_DWORD              dwTimeout       );
                                                                                            
ATECAT_API EC_T_DWORD  ecatSlaveSerializeMbxTfers(  EC_T_DWORD              dwSlaveId       );

ATECAT_API EC_T_DWORD  ecatSlaveParallelMbxTfers(   EC_T_DWORD              dwSlaveId       );

ATECAT_API EC_T_DWORD  ecatRegisterClient(          EC_PF_NOTIFY            pfnNotify,
                                                    EC_T_VOID*              pCallerData,
                                                    EC_T_REGISTERRESULTS*   pRegResults     );
ATECAT_API EC_T_DWORD  ecatUnregisterClient(        EC_T_DWORD              dwClntId        ); 
ATECAT_API EC_T_DWORD  ecatDcEnable(                EC_T_VOID                               );                             
ATECAT_API EC_T_DWORD  ecatDcDisable(               EC_T_VOID                               );                             
ATECAT_API EC_T_DWORD  ecatDcConfigure(             EC_T_DC_CONFIGURE*      pDcConfigure    );

ATECAT_API EC_T_DWORD  ecatGetSlaveInfo(            EC_T_BOOL               bFixedAddress,
                                                    EC_T_WORD               wSlaveAddress,
                                                    EC_T_GET_SLAVE_INFO*    pGetSlaveInfo   );

ATECAT_API EC_T_DWORD ecatGetSlaveInpVarInfoNumOf(  EC_T_BOOL               bFixedAddress,
                                                    EC_T_WORD               wSlaveAddress,
                                                    EC_T_WORD*              pwSlaveInpVarInfoNumOf);

ATECAT_API EC_T_DWORD ecatGetSlaveOutpVarInfoNumOf( EC_T_BOOL               bFixedAddress,
                                                    EC_T_WORD               wSlaveAddress,
                                                    EC_T_WORD*              pwSlaveOutpVarInfoNumOf );

ATECAT_API EC_T_DWORD ecatGetSlaveInpVarInfo(       EC_T_BOOL               bFixedAddress,
                                                    EC_T_WORD               wSlaveAddress,
                                                    EC_T_WORD               wNumOfVarsToRead,
                                                    EC_T_PROCESS_VAR_INFO*  pSlaveProcVarInfoEntries,
                                                    EC_T_WORD*              pwReadEntries    );


ATECAT_API EC_T_DWORD ecatGetSlaveOutpVarInfo(      EC_T_BOOL               bFixedAddress,
                                                    EC_T_WORD               wSlaveAddress,
                                                    EC_T_WORD               wNumOfVarsToRead,
                                                    EC_T_PROCESS_VAR_INFO*  pSlaveProcVarInfoEntries,
                                                    EC_T_WORD*              pwReadEntries );


ATECAT_API EC_T_DWORD ecatFindOutpVarByName(        EC_T_CHAR*              szVariableName,                 
                                                    EC_T_PROCESS_VAR_INFO*  pProcessVarInfoEntry);


ATECAT_API EC_T_DWORD ecatFindInpVarByName (        EC_T_CHAR*              szVariableName,
                                                    EC_T_PROCESS_VAR_INFO*  pSlaveOutpVarInfo );

ATECAT_API EC_T_DWORD  ecatEthDbgMsg(               EC_T_BYTE               byEthTypeByte0,
                                                    EC_T_BYTE               byEthTypeByte1,
                                                    EC_T_CHAR*              szMsg           );

ATECAT_API EC_T_DWORD   ecatBlockNode(              EC_T_SB_MISMATCH_DESC*  pMisMatch,
                                                    EC_T_DWORD              dwTimeout       );
ATECAT_API EC_T_DWORD   ecatOpenBlockedPorts(       EC_T_DWORD              dwTimeout       );
ATECAT_API EC_T_DWORD   ecatForceTopologyChange(    EC_T_VOID                               );
ATECAT_API EC_T_DWORD   ecatIsSlavePresent(         EC_T_DWORD              dwSlaveId,
                                                    EC_T_BOOL*              pbPresence      );






/* Multi instance API functions */

ATECAT_API EC_T_DWORD  emInitMaster(                EC_T_DWORD              dwInstanceID,
                                                    EC_T_INITMASTERPARMS*   pParms          );
ATECAT_API EC_T_DWORD  emDeinitMaster(              EC_T_DWORD              dwInstanceID);

ATECAT_API EC_T_DWORD  emStart(                     EC_T_DWORD              dwInstanceID,
                                                    EC_T_DWORD              dwTimeout       );
ATECAT_API EC_T_DWORD  emStop(                      EC_T_DWORD              dwInstanceID,
                                                    EC_T_DWORD              dwTimeout       );

ATECAT_API EC_T_DWORD  emIoControl(                 EC_T_DWORD              dwInstanceID,
                                                    EC_T_DWORD              dwCode, 
                                                    EC_T_IOCTLPARMS*        pParms          );

ATECAT_API EC_T_DWORD  emGetSlaveId(                EC_T_DWORD              dwInstanceID,
                                                    EC_T_WORD               wStationAddress );

ATECAT_API EC_T_DWORD  emGetSlaveFixedAddr(         EC_T_DWORD              dwInstanceID,           
                                                    EC_T_DWORD              dwSlaveId,              
                                                    EC_T_WORD*              pwFixedAddr     );

ATECAT_API EC_T_DWORD  emGetSlaveIdAtPosition(      EC_T_DWORD              dwInstanceID,
                                                    EC_T_WORD               wAutoIncAddress );
ATECAT_API EC_T_BOOL   emGetSlaveProp(              EC_T_DWORD              dwInstanceID,
                                                    EC_T_DWORD              dwSlaveId, 
                                                    EC_T_SLAVE_PROP*        pSlaveProp      );
ATECAT_API EC_T_DWORD  emGetSlaveState(             EC_T_DWORD              dwInstanceID,
                                                    EC_T_DWORD              dwSlaveId, 
                                                    EC_T_WORD*              pwCurrDevState, 
                                                    EC_T_WORD*              pwReqDevState, 
                                                    EC_T_DWORD              dwTimeout       );
ATECAT_API EC_T_DWORD  emSetSlaveState(             EC_T_DWORD              dwInstanceID,
                                                    EC_T_DWORD              dwSlaveId, 
                                                    EC_T_WORD               wNewReqDevState, 
                                                    EC_T_DWORD              dwTimeout       );
ATECAT_API EC_T_DWORD  emTferSingleRawCmd(          EC_T_DWORD              dwInstanceID,
                                                    EC_T_DWORD              dwSlaveId, 
                                                    EC_T_BYTE               byCmd, 
                                                    EC_T_DWORD              dwMemoryAddress, 
                                                    EC_T_VOID*              pvData, 
                                                    EC_T_WORD               wLen, 
                                                    EC_T_DWORD              dwTimeout       );
ATECAT_API EC_T_DWORD  emQueueRawCmd(               EC_T_DWORD              dwInstanceID,
                                                    EC_T_DWORD              dwSlaveId,
                                                    EC_T_WORD               wInvokeId,
                                                    EC_T_BYTE               byCmd,
                                                    EC_T_DWORD              dwMemoryAddress,
                                                    EC_T_VOID*              pvData,
                                                    EC_T_WORD               wLen            );
ATECAT_API EC_T_DWORD  emClntQueueRawCmd(           EC_T_DWORD              dwClntId,
                                                    EC_T_DWORD              dwInstanceID,
                                                    EC_T_DWORD              dwSlaveId,
                                                    EC_T_WORD               wInvokeId,
                                                    EC_T_BYTE               byCmd,
                                                    EC_T_DWORD              dwMemoryAddress,
                                                    EC_T_VOID*              pvData,
                                                    EC_T_WORD               wLen            );
ATECAT_API EC_T_DWORD  emGetNumConfiguredSlaves(    EC_T_DWORD              dwInstanceID);


ATECAT_API EC_T_MBXTFER* emMbxTferCreate(           EC_T_DWORD              dwInstanceID,
                                                    EC_T_MBXTFER_DESC*      pMbxTferDesc    );
ATECAT_API EC_T_VOID     emMbxTferDelete(           EC_T_DWORD              dwInstanceID,
                                                    EC_T_MBXTFER*           pMbxTfer        );

ATECAT_API EC_T_DWORD  emCoeSdoDownloadReq(         EC_T_DWORD              dwInstanceID,
                                                    EC_T_MBXTFER*           pMbxTfer, 
                                                    EC_T_DWORD              dwSlaveId, 
                                                    EC_T_WORD               wObIndex, 
                                                    EC_T_BYTE               byObSubIndex, 
                                                    EC_T_DWORD              dwTimeout,
                                                    EC_T_DWORD              dwFlags         );
ATECAT_API EC_T_DWORD   emCoeSdoDownload(           EC_T_DWORD              dwInstanceID
                                                   ,EC_T_DWORD              dwSlaveId
                                                   ,EC_T_WORD               wObIndex
                                                   ,EC_T_BYTE               byObSubIndex
                                                   ,EC_T_BYTE*              pbyData
                                                   ,EC_T_DWORD              dwDataLen
                                                   ,EC_T_DWORD              dwTimeout
                                                   ,EC_T_DWORD              dwFlags         );

ATECAT_API EC_T_DWORD  emCoeSdoUploadReq(           EC_T_DWORD              dwInstanceID,
                                                    EC_T_MBXTFER*           pMbxTfer, 
                                                    EC_T_DWORD              dwSlaveId, 
                                                    EC_T_WORD               wObIndex, 
                                                    EC_T_BYTE               byObSubIndex, 
                                                    EC_T_DWORD              dwTimeout,
                                                    EC_T_DWORD              dwFlags         );
ATECAT_API EC_T_DWORD  emCoeSdoUpload(              EC_T_DWORD              dwInstanceID
                                                   ,EC_T_DWORD              dwSlaveId
                                                   ,EC_T_WORD               wObIndex
                                                   ,EC_T_BYTE               byObSubIndex
                                                   ,EC_T_BYTE*              pbyData
                                                   ,EC_T_DWORD              dwDataLen
                                                   ,EC_T_DWORD*             pdwOutDataLen
                                                   ,EC_T_DWORD              dwTimeout
                                                   ,EC_T_DWORD              dwFlags         );
ATECAT_API EC_T_DWORD  emCoeGetODList(              EC_T_DWORD              dwInstanceID,
                                                    EC_T_MBXTFER*           pMbxTfer, 
                                                    EC_T_DWORD              dwSlaveId, 
                                                    EC_T_COE_ODLIST_TYPE    eListType, 
                                                    EC_T_DWORD              dwTimeout       );
ATECAT_API EC_T_DWORD  emCoeGetObjectDesc(          EC_T_DWORD              dwInstanceID,
                                                    EC_T_MBXTFER*           pMbxTfer, 
                                                    EC_T_DWORD              dwSlaveId, 
                                                    EC_T_WORD               wObIndex, 
                                                    EC_T_DWORD              dwTimeout       );
ATECAT_API EC_T_DWORD  emCoeGetEntryDesc(           EC_T_DWORD              dwInstanceID,
                                                    EC_T_MBXTFER*           pMbxTfer, 
                                                    EC_T_DWORD              dwSlaveId, 
                                                    EC_T_WORD               wObIndex, 
                                                    EC_T_BYTE               byObSubIndex, 
                                                    EC_T_BYTE               byValueInfo,
                                                    EC_T_DWORD              dwTimeout       );

#ifdef INCLUDE_COE_PDO_SUPPORT
ATECAT_API EC_T_DWORD emCoeRxPdoTfer(               EC_T_DWORD              dwInstanceID,
                                                    EC_T_MBXTFER*           pMbxTfer,
                                                    EC_T_DWORD              dwSlaveId,
                                                    EC_T_DWORD              dwNumber,
                                                    EC_T_DWORD              dwTimeout       );
#endif

ATECAT_API EC_T_DWORD  emFoeFileUpload(             EC_T_DWORD              dwInstanceID,
                                                    EC_T_DWORD              dwSlaveId,
                                                    EC_T_CHAR*              szFileName,
                                                    EC_T_DWORD              dwFileNameLen,
                                                    EC_T_BYTE*              pbyData,
                                                    EC_T_DWORD              dwDataLen,
                                                    EC_T_DWORD*             pdwOutDataLen,
                                                    EC_T_DWORD              dwPassWd,
                                                    EC_T_DWORD              dwTimeout       );

ATECAT_API EC_T_DWORD  emFoeFileDownload(           EC_T_DWORD              dwInstanceID,
                                                    EC_T_DWORD              dwSlaveId,
                                                    EC_T_CHAR*              szFileName,
                                                    EC_T_DWORD              dwFileNameLen,
                                                    EC_T_BYTE*              pbyData,
                                                    EC_T_DWORD              dwDataLen,
                                                    EC_T_DWORD              dwPassWd,
                                                    EC_T_DWORD              dwTimeout       );

ATECAT_API EC_T_DWORD  emFoeUploadReq(              EC_T_DWORD              dwInstanceID,
                                                    EC_T_MBXTFER*           pMbxTfer,
                                                    EC_T_DWORD              dwSlaveId,
                                                    EC_T_CHAR*              szFileName,
                                                    EC_T_DWORD              dwFileNameLen,
                                                    EC_T_DWORD              dwPassWd,
                                                    EC_T_DWORD              dwTimeout       );

ATECAT_API EC_T_DWORD  emFoeDownloadReq(            EC_T_DWORD              dwInstanceID,
                                                    EC_T_MBXTFER*           pMbxTfer,
                                                    EC_T_DWORD              dwSlaveId,
                                                    EC_T_CHAR*              szFileName,
                                                    EC_T_DWORD              dwFileNameLen,
                                                    EC_T_DWORD              dwPassWd,
                                                    EC_T_DWORD              dwTimeout       );

ATECAT_API EC_T_DWORD  emEoeRegisterEndpoint(       EC_T_DWORD              dwInstanceID,
                                                    EC_T_CHAR*              szEoEDrvIdent,
                                                    EC_T_VOID*              pLinkDrvDesc    ); 

#ifdef INCLUDE_SOE_SUPPORT
ATECAT_API EC_T_DWORD  emSoeWrite(                  EC_T_DWORD              dwInstanceID,            
                                                    EC_T_DWORD              dwSlaveId,
                                                    EC_T_BYTE               byDriveNo,
                                                    EC_T_BYTE               byElementFlags,
                                                    EC_T_WORD               wIDN,
                                                    EC_T_BYTE*              pbyData,
                                                    EC_T_DWORD              dwDataLen,
                                                    EC_T_DWORD              dwTimeout        );
ATECAT_API EC_T_DWORD  emSoeRead(                   EC_T_DWORD              dwInstanceID,               
                                                    EC_T_DWORD              dwSlaveId,
                                                    EC_T_BYTE               byDriveNo,
                                                    EC_T_BYTE               byElementFlags,
                                                    EC_T_WORD               wIDN, 
                                                    EC_T_BYTE*              pbyData,
                                                    EC_T_DWORD              dwDataLen,  
                                                    EC_T_DWORD*             pdwOutDataLen,
                                                    EC_T_DWORD              dwTimeout        );
ATECAT_API EC_T_DWORD  emSoeAbortProcCmd(           EC_T_DWORD              dwInstanceID,  
                                                    EC_T_DWORD              dwSlaveId,
                                                    EC_T_BYTE               byDriveNo,
                                                    EC_T_BYTE               byElementFlags,
                                                    EC_T_WORD               wIDN,
                                                    EC_T_DWORD              dwTimeout        );
#endif


ATECAT_API EC_T_DWORD  emConfigureMaster(           EC_T_DWORD              dwInstanceID,
                                                    EC_T_CNF_TYPE           eCnfType,
                                                    EC_T_PBYTE              pbyCnfData, 
                                                    EC_T_DWORD              dwCnfDataLen    );

ATECAT_API EC_T_DWORD  emSetMasterState(            EC_T_DWORD              dwInstanceID,
                                                    EC_T_DWORD              dwTimeout,
                                                    EC_T_STATE              eReqState       );
ATECAT_API EC_T_STATE  emGetMasterState(            EC_T_DWORD              dwInstanceID    );

ATECAT_API EC_T_DWORD  emExecJob(                   EC_T_DWORD              dwInstanceID,
                                                    EC_T_USER_JOB           eUserJob,
                                                    EC_T_PVOID              pvParam         );

ATECAT_API EC_T_DWORD  emGetProcessData(            EC_T_DWORD              dwInstanceID,
                                                    EC_T_BOOL               bOutputData,
                                                    EC_T_DWORD              dwOffset,
                                                    EC_T_BYTE*              pbyData,
                                                    EC_T_DWORD              dwLength,
                                                    EC_T_DWORD              dwTimeout       );

ATECAT_API EC_T_DWORD  emSetProcessData(            EC_T_DWORD              dwInstanceID,
                                                    EC_T_BOOL               bOutputData,
                                                    EC_T_DWORD              dwOffset,
                                                    EC_T_BYTE*              pbyData,
                                                    EC_T_DWORD              dwLength,
                                                    EC_T_DWORD              dwTimeout       );

ATECAT_API EC_T_DWORD  emGetNumConnectedSlaves(     EC_T_DWORD              dwInstanceID    );

ATECAT_API EC_T_DWORD  emReadSlaveEEPRom(           EC_T_DWORD              dwInstanceID,
                                                    EC_T_BOOL               bFixedAddressing,
                                                    EC_T_WORD               wSlaveAddress,
                                                    EC_T_WORD               wEEPRomStartOffset,
                                                    EC_T_WORD*              pwReadData,
                                                    EC_T_DWORD              dwReadLen,
                                                    EC_T_DWORD*             pdwNumOutData,
                                                    EC_T_DWORD              dwTimeout       );
ATECAT_API EC_T_DWORD  emWriteSlaveEEPRom(          EC_T_DWORD              dwInstanceID,
                                                    EC_T_BOOL               bFixedAddressing,
                                                    EC_T_WORD               wSlaveAddress,
                                                    EC_T_WORD               wEEPRomStartOffset,
                                                    EC_T_WORD*              pwWriteData,
                                                    EC_T_DWORD              dwWriteLen,
                                                    EC_T_DWORD              dwTimeout       );
ATECAT_API EC_T_DWORD emReloadSlaveEEPRom(          EC_T_DWORD              dwInstanceID,
                                                    EC_T_BOOL               bFixedAddressing,
                                                    EC_T_WORD               wSlaveAddress,
                                                    EC_T_DWORD              dwTimeout       );
ATECAT_API EC_T_DWORD emResetSlaveController(       EC_T_DWORD              dwInstanceID,
                                                    EC_T_BOOL               bFixedAddressing,
                                                    EC_T_WORD               wSlaveAddress,
                                                    EC_T_DWORD              dwTimeout       );
ATECAT_API EC_T_DWORD emAssignSlaveEEPRom(          EC_T_DWORD              dwInstanceID,
                                                    EC_T_BOOL               bFixedAddressing,
                                                    EC_T_WORD               wSlaveAddress,
                                                    EC_T_BOOL               bSlavePDIAccessEnable,
                                                    EC_T_BOOL               bForceAssign,
                                                    EC_T_DWORD              dwTimeout       );
ATECAT_API EC_T_DWORD emActiveSlaveEEPRom(          EC_T_DWORD              dwInstanceID,
                                                    EC_T_BOOL               bFixedAddressing,
                                                    EC_T_WORD               wSlaveAddress,
                                                    EC_T_BOOL*              pbSlavePDIAccessActive,
                                                    EC_T_DWORD              dwTimeout       );

ATECAT_API EC_T_DWORD emHCAcceptTopoChange(         EC_T_DWORD              dwInstanceID    );
ATECAT_API EC_T_DWORD emHCGetNumGroupMembers(       EC_T_DWORD              dwInstanceID,
                                                    EC_T_DWORD              dwGroupIndex    );
ATECAT_API EC_T_DWORD emHCGetSlaveIdsOfGroup(       EC_T_DWORD              dwInstanceID,
                                                    EC_T_DWORD              dwGroupIndex,
                                                    EC_T_DWORD*             adwSlaveId,
                                                    EC_T_DWORD              dwMaxNumSlaveIds );
ATECAT_API EC_T_DWORD emSetSlavePortState(          EC_T_DWORD              dwInstanceID,
                                                    EC_T_DWORD              dwSlaveID,
                                                    EC_T_WORD               wPort,
                                                    EC_T_BOOL               bClose,
                                                    EC_T_BOOL               bForce,
                                                    EC_T_DWORD              dwTimeout       );

ATECAT_API EC_T_DWORD  emSlaveSerializeMbxTfers(    EC_T_DWORD              dwInstanceID,
                                                    EC_T_DWORD              dwSlaveId       );

ATECAT_API EC_T_DWORD  emSlaveParallelMbxTfers(     EC_T_DWORD              dwInstanceID,
                                                    EC_T_DWORD              dwSlaveId       );

ATECAT_API EC_T_DWORD emRegisterClient(             EC_T_DWORD              dwInstanceID,    
                                                    EC_PF_NOTIFY            pfnNotify,
                                                    EC_T_VOID*              pCallerData,     
                                                    EC_T_REGISTERRESULTS*   pRegResults     );
ATECAT_API EC_T_DWORD emUnregisterClient(           EC_T_DWORD              dwInstanceID,
                                                    EC_T_DWORD              dwClntId        );                              
ATECAT_API EC_T_DWORD emDcEnable(                   EC_T_DWORD              dwInstanceID    );
ATECAT_API EC_T_DWORD emDcDisable(                  EC_T_DWORD              dwInstanceID    );
ATECAT_API EC_T_DWORD emDcConfigure(                EC_T_DWORD              dwInstanceID,
                                                    EC_T_DC_CONFIGURE*      pDcConfigure    );
ATECAT_API EC_T_DWORD emGetSlaveInfo(               EC_T_DWORD              dwInstanceID,
                                                    EC_T_BOOL               bFixedAddress,
                                                    EC_T_WORD               wSlaveAddress,
                                                    EC_T_GET_SLAVE_INFO*    pGetSlaveInfo   );

ATECAT_API EC_T_DWORD emGetSlaveInpVarInfoNumOf(    EC_T_DWORD              dwInstanceID,     
                                                    EC_T_BOOL               bFixedAddress,
                                                    EC_T_WORD               wSlaveAddress,
                                                    EC_T_WORD*              pwSlaveInpVarInfoNumOf);

ATECAT_API EC_T_DWORD emGetSlaveOutpVarInfoNumOf(   EC_T_DWORD              dwInstanceID,     
                                                    EC_T_BOOL               bFixedAddress,
                                                    EC_T_WORD               wSlaveAddress,
                                                    EC_T_WORD*              pwSlaveOutpVarInfoNumOf );

ATECAT_API EC_T_DWORD emGetSlaveInpVarInfo(         EC_T_DWORD              dwInstanceID,
                                                    EC_T_BOOL               bFixedAddress,
                                                    EC_T_WORD               wSlaveAddress,
                                                    EC_T_WORD               wNumOfVarsToRead,
                                                    EC_T_PROCESS_VAR_INFO*  pSlaveProcVarInfoEntries,
                                                    EC_T_WORD*              pwReadEntries    );


ATECAT_API EC_T_DWORD emGetSlaveOutpVarInfo(        EC_T_DWORD              dwInstanceID,
                                                    EC_T_BOOL               bFixedAddress,
                                                    EC_T_WORD               wSlaveAddress,
                                                    EC_T_WORD               wNumOfVarsToRead,
                                                    EC_T_PROCESS_VAR_INFO*  pSlaveProcVarInfoEntries,
                                                    EC_T_WORD*              pwReadEntries    );


ATECAT_API EC_T_DWORD emFindOutpVarByName  (        EC_T_DWORD              dwInstanceID,
                                                    EC_T_CHAR*              szVariableName,
                                                    EC_T_PROCESS_VAR_INFO*  pSlaveOutpVarInfo );


ATECAT_API EC_T_DWORD emFindInpVarByName  (         EC_T_DWORD              dwInstanceID,
                                                    EC_T_CHAR*              szVariableName,
                                                    EC_T_PROCESS_VAR_INFO*  pSlaveOutpVarInfo );

ATECAT_API EC_T_DWORD emEthDbgMsg(                  EC_T_DWORD              dwInstanceID,
                                                    EC_T_BYTE               byEthTypeByte0,
                                                    EC_T_BYTE               byEthTypeByte1,
                                                    EC_T_CHAR*              szMsg           );
ATECAT_API EC_T_DWORD emBlockNode(                  EC_T_DWORD              dwInstanceID,
                                                    EC_T_SB_MISMATCH_DESC*  pMisMatch,
                                                    EC_T_DWORD              dwTimeout       );
ATECAT_API EC_T_DWORD emOpenBlockedPorts(           EC_T_DWORD              dwInstanceID,
                                                    EC_T_DWORD              dwTimeout       );
ATECAT_API EC_T_DWORD emForceTopologyChange(        EC_T_DWORD              dwInstanceID    );
ATECAT_API EC_T_DWORD emIsSlavePresent(             EC_T_DWORD              dwInstanceID,
                                                    EC_T_DWORD              dwSlaveId,
                                                    EC_T_BOOL*              pbPresence      );


#ifdef __cplusplus
} /* extern "C" */
#endif

#endif /* INC_ATETHERCAT */

/*-END OF SOURCE FILE--------------------------------------------------------*/
